"use client";
import { useEffect, useState, FormEvent } from "react";
import Link from "next/link";
import { useSession } from "next-auth/react";
import { useLanguage } from "@/contexts/LanguageContext";
import {
  ArrowLeft,
  Users,
  Loader2,
  Plus,
  Pencil,
  Trash2,
  Check,
  X,
  Search,
  Filter,
  Eye,
  EyeOff,
} from "lucide-react";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";

type Role = {
  id: string;
  name: string;
  description?: string | null;
  isActive: boolean;
  createdAt: string;
  _count: {
    users: number;
  };
};

export default function AdminRoles() {
  const { data: session, status } = useSession();
  const { t } = useLanguage();
  const role = String((session?.user as any)?.role || "").toUpperCase();
  const isAdmin = role === "ADMIN";

  const [roles, setRoles] = useState<Role[]>([]);
  const [loading, setLoading] = useState(true);
  const [err, setErr] = useState<string | null>(null);

  // Filter states
  const [filters, setFilters] = useState({
    search: "",
    isActive: "",
  });
  const [showFilters, setShowFilters] = useState(false);

  // Create form
  const [name, setName] = useState("");
  const [description, setDescription] = useState("");
  const [creating, setCreating] = useState(false);

  // Edit form
  const [editingId, setEditingId] = useState<string | null>(null);
  const [editName, setEditName] = useState("");
  const [editDescription, setEditDescription] = useState("");
  const [saving, setSaving] = useState(false);

  // Confirmation dialog
  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
  const [roleToDelete, setRoleToDelete] = useState<Role | null>(null);

  // Feedback
  const [msg, setMsg] = useState<string | null>(null);

  const fetchRoles = () => {
    setLoading(true);
    setErr(null);

    // Build query string from filters
    const queryParams = new URLSearchParams();
    if (filters.search) queryParams.append("search", filters.search);
    if (filters.isActive) queryParams.append("isActive", filters.isActive);

    const queryString = queryParams.toString();
    const url = queryString
      ? `/api/admin/roles?${queryString}`
      : "/api/admin/roles";

    fetch(url)
      .then((r) => r.json())
      .then((j) => {
        if (!j.ok) throw new Error(j.error || "Failed to load roles");
        setRoles(j.roles || []);
      })
      .catch((e) => setErr(String(e)))
      .finally(() => setLoading(false));
  };

  useEffect(() => {
    fetchRoles();
  }, [filters]);

  const onCreate = async (e: FormEvent) => {
    e.preventDefault();
    setErr(null);
    setMsg(null);
    setCreating(true);

    try {
      const resp = await fetch("/api/admin/roles", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          name: name.trim(),
          description: description.trim() || undefined,
        }),
      });
      const json = await resp.json();
      if (!json.ok) throw new Error(json.error || "Create failed");

      setMsg("Role created successfully.");
      setName("");
      setDescription("");
      fetchRoles();
    } catch (e: any) {
      setErr(String(e?.message || e));
    } finally {
      setCreating(false);
    }
  };

  const onEdit = (role: Role) => {
    setEditingId(role.id);
    setEditName(role.name);
    setEditDescription(role.description || "");
  };

  const onSaveEdit = async () => {
    if (!editingId) return;

    setSaving(true);
    setErr(null);
    setMsg(null);

    try {
      const resp = await fetch(`/api/admin/roles/${editingId}`, {
        method: "PATCH",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          name: editName.trim(),
          description: editDescription.trim() || undefined,
        }),
      });
      const json = await resp.json();
      if (!json.ok) throw new Error(json.error || "Update failed");

      setMsg("Role updated successfully.");
      setEditingId(null);
      fetchRoles();
    } catch (e: any) {
      setErr(String(e?.message || e));
    } finally {
      setSaving(false);
    }
  };

  const onToggleActive = async (role: Role) => {
    try {
      const resp = await fetch(`/api/admin/roles/${role.id}`, {
        method: "PATCH",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ isActive: !role.isActive }),
      });
      const json = await resp.json();
      if (!json.ok) throw new Error(json.error || "Toggle failed");

      setMsg(
        `Role ${!role.isActive ? "activated" : "deactivated"} successfully.`
      );
      fetchRoles();
    } catch (e: any) {
      setErr(String(e?.message || e));
    }
  };

  const onDelete = (role: Role) => {
    setRoleToDelete(role);
    setDeleteDialogOpen(true);
  };

  const handleDeleteConfirm = async () => {
    if (!roleToDelete) return;

    setErr(null);
    setMsg(null);

    try {
      const resp = await fetch(`/api/admin/roles/${roleToDelete.id}`, {
        method: "DELETE",
      });
      const json = await resp.json();
      if (!json.ok) throw new Error(json.error || "Delete failed");

      setMsg("Role deleted successfully.");
      setDeleteDialogOpen(false);
      setRoleToDelete(null);
      fetchRoles();
    } catch (e: any) {
      setErr(String(e?.message || e));
    }
  };

  const clearFilters = () => {
    setFilters({ search: "", isActive: "" });
  };

  const hasActiveFilters = Object.values(filters).some((value) => value !== "");

  if (status === "loading") {
    return (
      <main className='max-w-6xl mx-auto px-4 py-6'>{t("common.loading")}</main>
    );
  }

  if (!isAdmin) {
    return (
      <main className='max-w-6xl mx-auto px-4 py-6'>
        <h1 className='text-2xl font-bold'>
          {t("admin.manageRoles") || "Manage Roles"}
        </h1>
        <div className='alert alert-error mt-3'>
          <X />
          <div>
            <p className='font-semibold'>
              {t("admin.forbidden") || "Forbidden — admin only"}
            </p>
            <p className='text-sm opacity-90'>
              {t("admin.noPermission") ||
                "You don't have permission to view this page."}
            </p>
          </div>
        </div>
      </main>
    );
  }

  return (
    <main className='max-w-6xl mx-auto px-4 py-6'>
      {/* Header */}
      <header className='flex items-center gap-3 mb-4'>
        <Link
          href='/admin/technicians'
          className='btn-secondary inline-flex items-center gap-2'
        >
          <ArrowLeft className='w-4 h-4' />
          {t("common.back")} {t("common.to")} {t("admin.technicians")}
        </Link>
        <h1 className='text-2xl font-bold ml-1 flex items-center gap-2'>
          <Users className='w-5 h-5 text-[color:var(--color-primary)]' />
          {t("admin.manageRoles") || "Manage Roles"}
        </h1>
      </header>

      {/* Feedback */}
      {msg && (
        <div className='alert alert-success mb-3'>
          <Check />
          <div>
            <p className='font-semibold'>{t("common.success")}</p>
            <p className='text-sm opacity-90'>{msg}</p>
          </div>
        </div>
      )}
      {err && (
        <div className='alert alert-error mb-3'>
          <X />
          <div>
            <p className='font-semibold'>{t("common.error")}</p>
            <p className='text-sm opacity-90'>{err}</p>
          </div>
        </div>
      )}

      {/* Create Role */}
      <section className='card p-5 mb-6'>
        <h2 className='text-lg font-semibold mb-4'>
          {t("admin.create")} {t("sidebar.roles")}
        </h2>
        <form onSubmit={onCreate} className='grid gap-4 max-w-xl'>
          <div>
            <label className='block text-sm font-medium mb-1'>
              {t("sidebar.roles")} {t("common.name") || "Name"}
            </label>
            <input
              type='text'
              required
              value={name}
              onChange={(e) => setName(e.target.value)}
              className='input-field'
              placeholder='e.g., Network Specialist, Database Admin'
            />
          </div>

          <div>
            <label className='block text-sm font-medium mb-1'>
              {t("admin.description") || "Description"}{" "}
              <span className='text-[color:var(--color-muted)]'>
                ({t("common.optional") || "optional"})
              </span>
            </label>
            <textarea
              value={description}
              onChange={(e) => setDescription(e.target.value)}
              className='input-field'
              placeholder='Brief description of this role...'
              rows={3}
            />
          </div>

          <div className='pt-1'>
            <button
              type='submit'
              disabled={creating}
              className='btn-primary inline-flex items-center gap-2 disabled:opacity-60'
            >
              {creating ? (
                <>
                  <Loader2 className='w-4 h-4 animate-spin' />
                  {t("admin.creatingTechnician")}
                </>
              ) : (
                <>
                  <Plus className='w-4 h-4' />
                  {t("admin.create")} {t("sidebar.roles")}
                </>
              )}
            </button>
          </div>
        </form>
      </section>

      {/* List Roles */}
      <section className='card p-5'>
        <div className='flex items-center justify-between gap-3 mb-3'>
          <h2 className='text-lg font-semibold'>{t("sidebar.roles")}</h2>
          <div className='flex items-center gap-2'>
            <button
              onClick={() => setShowFilters(!showFilters)}
              className={`btn-secondary inline-flex items-center gap-2 ${
                showFilters || hasActiveFilters
                  ? "bg-primary-500 text-primary-fore"
                  : ""
              }`}
            >
              <Filter className='w-4 h-4' />
              {t("tickets.filters")}
            </button>
          </div>
        </div>

        {/* Filter Panel */}
        {showFilters && (
          <div className='mb-4 p-4 bg-sidebar-accent rounded-lg border border-default'>
            <div className='grid grid-cols-1 md:grid-cols-2 gap-4'>
              {/* Search */}
              <div>
                <label className='block text-sm font-medium mb-2'>
                  {t("common.search")}
                </label>
                <div className='relative'>
                  <Search className='absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-foreground/50' />
                  <input
                    type='text'
                    placeholder={t("admin.searchRoles") || "Search roles..."}
                    value={filters.search}
                    onChange={(e) =>
                      setFilters((prev) => ({
                        ...prev,
                        search: e.target.value,
                      }))
                    }
                    className='w-full pl-10 pr-3 py-2 border border-default rounded-md bg-background text-foreground focus:outline-none focus:ring-2 focus:ring-primary-500'
                  />
                </div>
              </div>

              {/* Status */}
              <div>
                <label className='block text-sm font-medium mb-2'>
                  {t("tickets.statusLabel")}
                </label>
                <select
                  value={filters.isActive}
                  onChange={(e) =>
                    setFilters((prev) => ({
                      ...prev,
                      isActive: e.target.value,
                    }))
                  }
                  className='w-full px-3 py-2 border border-default rounded-md bg-background text-foreground focus:outline-none focus:ring-2 focus:ring-primary-500'
                >
                  <option value=''>{t("admin.allStatus")}</option>
                  <option value='true'>{t("admin.active")}</option>
                  <option value='false'>{t("admin.inactive")}</option>
                </select>
              </div>
            </div>

            {/* Filter Actions */}
            <div className='flex items-center justify-between mt-4 pt-4 border-t border-default'>
              <div className='text-sm text-foreground/70'>
                {hasActiveFilters
                  ? `${roles.length} ${t("admin.rolesFound")}`
                  : t("admin.allRoles")}
              </div>
              <div className='flex items-center gap-2'>
                {hasActiveFilters && (
                  <button
                    onClick={clearFilters}
                    className='flex items-center gap-1 px-3 py-1 text-sm text-foreground/70 hover:text-foreground transition'
                  >
                    <X className='w-4 h-4' />
                    {t("tickets.clearFilters")}
                  </button>
                )}
              </div>
            </div>
          </div>
        )}

        {loading ? (
          <div className='flex items-center gap-2 text-[color:var(--color-muted)]'>
            <Loader2 className='w-4 h-4 animate-spin' /> {t("common.loading")}
          </div>
        ) : (
          <div
            className='overflow-x-auto rounded-lg border'
            style={{ borderColor: "var(--color-border)" }}
          >
            <table className='w-full text-sm'>
              <thead style={{ background: "var(--color-sidebar-accent)" }}>
                <tr>
                  <th className='text-left font-semibold px-4 py-2'>
                    {t("common.name")}
                  </th>
                  <th className='text-left font-semibold px-4 py-2'>
                    {t("admin.description")}
                  </th>
                  <th className='text-left font-semibold px-4 py-2'>
                    {t("admin.users") || "Users"}
                  </th>
                  <th className='text-left font-semibold px-4 py-2'>
                    {t("tickets.statusLabel")}
                  </th>
                  <th className='text-left font-semibold px-4 py-2'>
                    {t("tickets.table.created")}
                  </th>
                  <th className='text-left font-semibold px-4 py-2'>
                    {t("common.actions") || "Actions"}
                  </th>
                </tr>
              </thead>
              <tbody>
                {roles.map((role) => (
                  <tr
                    key={role.id}
                    className='border-t'
                    style={{ borderColor: "var(--color-border)" }}
                  >
                    <td className='px-4 py-2'>
                      {editingId === role.id ? (
                        <input
                          type='text'
                          value={editName}
                          onChange={(e) => setEditName(e.target.value)}
                          className='input-field'
                          disabled={saving}
                        />
                      ) : (
                        <span className='font-medium'>{role.name}</span>
                      )}
                    </td>
                    <td className='px-4 py-2'>
                      {editingId === role.id ? (
                        <textarea
                          value={editDescription}
                          onChange={(e) => setEditDescription(e.target.value)}
                          className='input-field'
                          rows={2}
                          disabled={saving}
                        />
                      ) : (
                        <span className='text-[color:var(--color-muted)]'>
                          {role.description || "—"}
                        </span>
                      )}
                    </td>
                    <td className='px-4 py-2'>
                      <span className='text-[color:var(--color-muted)]'>
                        {role._count.users} {t("admin.user") || "user"}
                        {role._count.users !== 1 ? "s" : ""}
                      </span>
                    </td>
                    <td className='px-4 py-2'>
                      <span
                        className={`inline-flex items-center gap-1 px-2 py-0.5 rounded-full text-xs font-medium ${
                          role.isActive
                            ? "bg-green-100 text-green-800"
                            : "bg-red-100 text-red-800"
                        }`}
                      >
                        {role.isActive ? (
                          <>
                            <Eye className='w-3 h-3' />
                            {t("admin.active") || "Active"}
                          </>
                        ) : (
                          <>
                            <EyeOff className='w-3 h-3' />
                            {t("admin.inactive") || "Inactive"}
                          </>
                        )}
                      </span>
                    </td>
                    <td className='px-4 py-2 text-[color:var(--color-muted)]'>
                      {new Date(role.createdAt).toLocaleDateString()}
                    </td>
                    <td className='px-4 py-2'>
                      <div className='flex items-center gap-2'>
                        {editingId === role.id ? (
                          <>
                            <button
                              onClick={onSaveEdit}
                              disabled={saving}
                              className='btn-primary text-xs px-2 py-1 disabled:opacity-60'
                            >
                              {saving ? (
                                <Loader2 className='w-3 h-3 animate-spin' />
                              ) : (
                                <Check className='w-3 h-3' />
                              )}
                            </button>
                            <button
                              onClick={() => setEditingId(null)}
                              disabled={saving}
                              className='btn-secondary text-xs px-2 py-1 disabled:opacity-60'
                            >
                              <X className='w-3 h-3' />
                            </button>
                          </>
                        ) : (
                          <>
                            <button
                              onClick={() => onEdit(role)}
                              className='btn-secondary text-xs px-2 py-1'
                              title='Edit role'
                            >
                              <Pencil className='w-3 h-3' />
                            </button>
                            <button
                              onClick={() => onToggleActive(role)}
                              className='btn-secondary text-xs px-2 py-1'
                              title={
                                role.isActive
                                  ? "Deactivate role"
                                  : "Activate role"
                              }
                            >
                              {role.isActive ? (
                                <EyeOff className='w-3 h-3' />
                              ) : (
                                <Eye className='w-3 h-3' />
                              )}
                            </button>
                            <button
                              onClick={() => onDelete(role)}
                              disabled={role._count.users > 0}
                              className='btn-secondary text-xs px-2 py-1 disabled:opacity-60'
                              title={
                                role._count.users > 0
                                  ? "Cannot delete role with assigned users"
                                  : "Delete role"
                              }
                            >
                              <Trash2 className='w-3 h-3' />
                            </button>
                          </>
                        )}
                      </div>
                    </td>
                  </tr>
                ))}
                {roles.length === 0 && (
                  <tr>
                    <td
                      className='px-4 py-6 text-center text-[color:var(--color-muted)]'
                      colSpan={6}
                    >
                      No roles found.
                    </td>
                  </tr>
                )}
              </tbody>
            </table>
          </div>
        )}
      </section>

      <Dialog open={deleteDialogOpen} onOpenChange={setDeleteDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Delete Role</DialogTitle>
            <DialogDescription>
              Are you sure you want to delete the role "{roleToDelete?.name}"?
              This action cannot be undone.
            </DialogDescription>
          </DialogHeader>
          <DialogFooter>
            <button
              onClick={() => {
                setDeleteDialogOpen(false);
                setRoleToDelete(null);
              }}
              className='btn-secondary'
            >
              Cancel
            </button>
            <button onClick={handleDeleteConfirm} className='btn-danger'>
              Delete
            </button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </main>
  );
}
